﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>
#include <aws/rds/model/Tag.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 * <p/><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/rds-2014-10-31/CreateEventSubscriptionMessage">AWS
 * API Reference</a></p>
 */
class CreateEventSubscriptionRequest : public RDSRequest {
 public:
  AWS_RDS_API CreateEventSubscriptionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateEventSubscription"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name of the subscription.</p> <p>Constraints: The name must be less than
   * 255 characters.</p>
   */
  inline const Aws::String& GetSubscriptionName() const { return m_subscriptionName; }
  inline bool SubscriptionNameHasBeenSet() const { return m_subscriptionNameHasBeenSet; }
  template <typename SubscriptionNameT = Aws::String>
  void SetSubscriptionName(SubscriptionNameT&& value) {
    m_subscriptionNameHasBeenSet = true;
    m_subscriptionName = std::forward<SubscriptionNameT>(value);
  }
  template <typename SubscriptionNameT = Aws::String>
  CreateEventSubscriptionRequest& WithSubscriptionName(SubscriptionNameT&& value) {
    SetSubscriptionName(std::forward<SubscriptionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the SNS topic created for event
   * notification. SNS automatically creates the ARN when you create a topic and
   * subscribe to it.</p>  <p>RDS doesn't support FIFO (first in, first out)
   * topics. For more information, see <a
   * href="https://docs.aws.amazon.com/sns/latest/dg/sns-fifo-topics.html">Message
   * ordering and deduplication (FIFO topics)</a> in the <i>Amazon Simple
   * Notification Service Developer Guide</i>.</p>
   */
  inline const Aws::String& GetSnsTopicArn() const { return m_snsTopicArn; }
  inline bool SnsTopicArnHasBeenSet() const { return m_snsTopicArnHasBeenSet; }
  template <typename SnsTopicArnT = Aws::String>
  void SetSnsTopicArn(SnsTopicArnT&& value) {
    m_snsTopicArnHasBeenSet = true;
    m_snsTopicArn = std::forward<SnsTopicArnT>(value);
  }
  template <typename SnsTopicArnT = Aws::String>
  CreateEventSubscriptionRequest& WithSnsTopicArn(SnsTopicArnT&& value) {
    SetSnsTopicArn(std::forward<SnsTopicArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of source that is generating the events. For example, if you want to
   * be notified of events generated by a DB instance, you set this parameter to
   * <code>db-instance</code>. For RDS Proxy events, specify <code>db-proxy</code>.
   * If this value isn't specified, all events are returned.</p> <p>Valid
   * Values:<code> db-instance | db-cluster | db-parameter-group | db-security-group
   * | db-snapshot | db-cluster-snapshot | db-proxy | zero-etl |
   * custom-engine-version | blue-green-deployment </code> </p>
   */
  inline const Aws::String& GetSourceType() const { return m_sourceType; }
  inline bool SourceTypeHasBeenSet() const { return m_sourceTypeHasBeenSet; }
  template <typename SourceTypeT = Aws::String>
  void SetSourceType(SourceTypeT&& value) {
    m_sourceTypeHasBeenSet = true;
    m_sourceType = std::forward<SourceTypeT>(value);
  }
  template <typename SourceTypeT = Aws::String>
  CreateEventSubscriptionRequest& WithSourceType(SourceTypeT&& value) {
    SetSourceType(std::forward<SourceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of event categories for a particular source type
   * (<code>SourceType</code>) that you want to subscribe to. You can see a list of
   * the categories for a given source type in the "Amazon RDS event categories and
   * event messages" section of the <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_Events.Messages.html">
   * <i>Amazon RDS User Guide</i> </a> or the <a
   * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_Events.Messages.html">
   * <i>Amazon Aurora User Guide</i> </a>. You can also see this list by using the
   * <code>DescribeEventCategories</code> operation.</p>
   */
  inline const Aws::Vector<Aws::String>& GetEventCategories() const { return m_eventCategories; }
  inline bool EventCategoriesHasBeenSet() const { return m_eventCategoriesHasBeenSet; }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  void SetEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories = std::forward<EventCategoriesT>(value);
  }
  template <typename EventCategoriesT = Aws::Vector<Aws::String>>
  CreateEventSubscriptionRequest& WithEventCategories(EventCategoriesT&& value) {
    SetEventCategories(std::forward<EventCategoriesT>(value));
    return *this;
  }
  template <typename EventCategoriesT = Aws::String>
  CreateEventSubscriptionRequest& AddEventCategories(EventCategoriesT&& value) {
    m_eventCategoriesHasBeenSet = true;
    m_eventCategories.emplace_back(std::forward<EventCategoriesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The list of identifiers of the event sources for which events are returned.
   * If not specified, then all sources are included in the response. An identifier
   * must begin with a letter and must contain only ASCII letters, digits, and
   * hyphens. It can't end with a hyphen or contain two consecutive hyphens.</p>
   * <p>Constraints:</p> <ul> <li> <p>If <code>SourceIds</code> are supplied,
   * <code>SourceType</code> must also be provided.</p> </li> <li> <p>If the source
   * type is a DB instance, a <code>DBInstanceIdentifier</code> value must be
   * supplied.</p> </li> <li> <p>If the source type is a DB cluster, a
   * <code>DBClusterIdentifier</code> value must be supplied.</p> </li> <li> <p>If
   * the source type is a DB parameter group, a <code>DBParameterGroupName</code>
   * value must be supplied.</p> </li> <li> <p>If the source type is a DB security
   * group, a <code>DBSecurityGroupName</code> value must be supplied.</p> </li> <li>
   * <p>If the source type is a DB snapshot, a <code>DBSnapshotIdentifier</code>
   * value must be supplied.</p> </li> <li> <p>If the source type is a DB cluster
   * snapshot, a <code>DBClusterSnapshotIdentifier</code> value must be supplied.</p>
   * </li> <li> <p>If the source type is an RDS Proxy, a <code>DBProxyName</code>
   * value must be supplied.</p> </li> </ul>
   */
  inline const Aws::Vector<Aws::String>& GetSourceIds() const { return m_sourceIds; }
  inline bool SourceIdsHasBeenSet() const { return m_sourceIdsHasBeenSet; }
  template <typename SourceIdsT = Aws::Vector<Aws::String>>
  void SetSourceIds(SourceIdsT&& value) {
    m_sourceIdsHasBeenSet = true;
    m_sourceIds = std::forward<SourceIdsT>(value);
  }
  template <typename SourceIdsT = Aws::Vector<Aws::String>>
  CreateEventSubscriptionRequest& WithSourceIds(SourceIdsT&& value) {
    SetSourceIds(std::forward<SourceIdsT>(value));
    return *this;
  }
  template <typename SourceIdsT = Aws::String>
  CreateEventSubscriptionRequest& AddSourceIds(SourceIdsT&& value) {
    m_sourceIdsHasBeenSet = true;
    m_sourceIds.emplace_back(std::forward<SourceIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether to activate the subscription. If the event notification
   * subscription isn't activated, the subscription is created but not active.</p>
   */
  inline bool GetEnabled() const { return m_enabled; }
  inline bool EnabledHasBeenSet() const { return m_enabledHasBeenSet; }
  inline void SetEnabled(bool value) {
    m_enabledHasBeenSet = true;
    m_enabled = value;
  }
  inline CreateEventSubscriptionRequest& WithEnabled(bool value) {
    SetEnabled(value);
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateEventSubscriptionRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateEventSubscriptionRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_subscriptionName;

  Aws::String m_snsTopicArn;

  Aws::String m_sourceType;

  Aws::Vector<Aws::String> m_eventCategories;

  Aws::Vector<Aws::String> m_sourceIds;

  bool m_enabled{false};

  Aws::Vector<Tag> m_tags;
  bool m_subscriptionNameHasBeenSet = false;
  bool m_snsTopicArnHasBeenSet = false;
  bool m_sourceTypeHasBeenSet = false;
  bool m_eventCategoriesHasBeenSet = false;
  bool m_sourceIdsHasBeenSet = false;
  bool m_enabledHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
