/* SPDX-License-Identifier: GPL-3.0-or-later */
/* SPDX-FileCopyrightText: 2024 Riku Viitanen <riku.viitanen@protonmail.com> */

#include <stddef.h>

#include <getopt.h>
#include <libx86.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define EXH(r32) r32>>16
#define XX(r32) r32&0xffff
#define XH(r32) (r32>>8)&0xff
#define XL(r32) r32&0xff


void print_usage(char *pgmname) {
	printf("usage: %s interrupt [-a eax] [-b ebx] [-c ecx] [-d edx]\n", pgmname);
}

void pretty_1688_name(uint32_t r, uint32_t r2, char **pad, char name) {
	if (EXH(r) | EXH(r2)) {
		printf("%s   e%cx" , *pad, name);
		*pad = "     ";

	} else if (((XH(r) | XH(r2)) != 0) & ((XL(r) | XL(r2)) !=0)) {
		printf("%s %cx", *pad, name);
		*pad = "   ";

	} else if (XH(r) | XH(r2)) {
		printf("%s%ch", *pad, name);
		*pad = "  ";


	} else if (XL(r) | XL(r2)) {
		printf("%s%cl", *pad, name);
		*pad = "  ";
	}
}

void pretty_1688(uint32_t r, uint32_t r2, char **pad) {
	if (EXH(r) | EXH(r2)) {
		printf("%s%04x %04x" , *pad, EXH(r), XX(r));
		*pad = "  ";

	} else if (((XH(r) | XH(r2)) != 0) & ((XL(r) | XL(r2)) !=0)) {
		printf("%s%04x", *pad, XX(r));
		*pad = "  ";

	} else if (XH(r) | XH(r2)) {
		printf("%s%02x", *pad, XH(r));
		*pad = "  ";

	} else if (XL(r) | XL(r2)) {
		printf("%s%02x", *pad, XL(r));
		*pad = "  ";
	}
}

void pretty_1616_name(uint32_t r, uint32_t r2, char **pad, char *name) {
	if (EXH(r) | EXH(r2)) {
		printf("%s   e%2s", *pad, name);
		*pad = "     ";
	} else if (XX(r) | XX(r2)) {
		printf("%s %2s", *pad, name);
		*pad = " ";
	}
}

void pretty_1616(uint32_t r, uint32_t r2, char **pad) {
	if (EXH(r) | EXH(r2)) {
		printf("%s%04x %04x", *pad, EXH(r), XX(r));
		*pad = "  ";
	} else if (XX(r) | XX(r2)) {
		printf("%s%04x", *pad, XX(r));
		*pad = "  ";
	}
}

void pretty_16_name(uint16_t r, uint16_t r2, char **pad, char *name) {
	if (r | r2) {
		printf("%s %2s", *pad, name);
		*pad = "   ";
	}
}

void pretty_16(uint16_t r, uint16_t r2, char **pad) {
	if (r | r2) {
		printf("%s%04x", *pad, r);
		*pad = "  ";
	}
}


void print_pretty_regs(struct LRMI_regs *r, struct LRMI_regs *prev) {
	char *padding = "        ";
	pretty_1688_name(prev->eax, r->eax, &padding, 'a');
	pretty_1688_name(prev->ebx, r->ebx, &padding, 'b');
	pretty_1688_name(prev->ecx, r->ecx, &padding, 'c');
	pretty_1688_name(prev->edx, r->edx, &padding, 'd');
	pretty_16_name(prev->es, r->es, &padding, "es");
	pretty_1616_name(prev->edi, r->edi, &padding, "di");
	pretty_16_name(prev->ds, r->ds, &padding, "ds");
	pretty_1616_name(prev->esi, r->esi, &padding, "si");
	pretty_16_name(prev->fs, r->fs, &padding, "fs");
	pretty_16_name(prev->gs, r->gs, &padding, "gs");
	putchar('\n');

	printf("Before:");
	padding = " ";
	pretty_1688(prev->eax, r->eax, &padding);
	pretty_1688(prev->ebx, r->ebx, &padding);
	pretty_1688(prev->ecx, r->ecx, &padding);
	pretty_1688(prev->edx, r->edx, &padding);
	pretty_16(prev->es, r->es, &padding);
	pretty_1616(prev->edi, r->edi, &padding);
	pretty_16(prev->ds, r->ds, &padding);
	pretty_1616(prev->esi, r->esi, &padding);
	pretty_16(prev->fs, r->fs, &padding);
	pretty_16(prev->gs, r->gs, &padding);
	putchar('\n');

	printf("After:");
	padding = "  ";
	pretty_1688(r->eax, prev->eax, &padding);
	pretty_1688(r->ebx, prev->ebx, &padding);
	pretty_1688(r->ecx, prev->ecx, &padding);
	pretty_1688(r->edx, prev->edx, &padding);
	pretty_16(r->es, prev->es, &padding);
	pretty_1616(r->edi, prev->edi, &padding);
	pretty_16(r->ds, prev->ds, &padding);
	pretty_1616(r->esi, prev->edi, &padding);
	pretty_16(r->fs, prev->fs, &padding);
	pretty_16(r->gs, prev->gs, &padding);
	printf("\n");
}

int main(int argc, char **argv) {
	struct LRMI_regs r;
	memset(&r, 0, sizeof(r));
	if (!LRMI_init()) {
		fprintf(stderr, "Error: LRMI init failed!\n");
		return 1;
	}

	int interrupt;
	if (sscanf(argv[1], "%2x", &interrupt) != 1) {
		fprintf(stderr, "Error: missing interrupt number\n");
		print_usage(*argv);
		exit(1);
	}

	while (1) {
		int o = getopt(argc, argv, "a:b:c:d:");
		if (o == -1) break;
		switch (o) {
		case 'a':
			sscanf(optarg, "%8x", &(r.eax));
			break;
		case 'b':
			sscanf(optarg, "%8x", &(r.ebx));
			break;
		case 'c':
			sscanf(optarg, "%8x", &(r.ecx));
			break;
		case 'd':
			sscanf(optarg, "%8x", &(r.edx));
			break;
		default:
			printf("for help: %s -h\n", *argv);
			exit(2);
		}
	}

	struct LRMI_regs prev = r;
	struct LRMI_regs diff;

	printf("Calling int%02xh\n", interrupt);

	if (!LRMI_int(interrupt, &r)) {
		fprintf(stderr, "Error: LRMI call failed!\n");
		exit(4);
	}

	print_pretty_regs(&r, &prev);

	return 0;
}
