﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/events/CloudWatchEventsRequest.h>
#include <aws/events/CloudWatchEvents_EXPORTS.h>

#include <utility>

namespace Aws {
namespace CloudWatchEvents {
namespace Model {

/**
 */
class RemovePermissionRequest : public CloudWatchEventsRequest {
 public:
  AWS_CLOUDWATCHEVENTS_API RemovePermissionRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "RemovePermission"; }

  AWS_CLOUDWATCHEVENTS_API Aws::String SerializePayload() const override;

  AWS_CLOUDWATCHEVENTS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The statement ID corresponding to the account that is no longer allowed to
   * put events to the default event bus.</p>
   */
  inline const Aws::String& GetStatementId() const { return m_statementId; }
  inline bool StatementIdHasBeenSet() const { return m_statementIdHasBeenSet; }
  template <typename StatementIdT = Aws::String>
  void SetStatementId(StatementIdT&& value) {
    m_statementIdHasBeenSet = true;
    m_statementId = std::forward<StatementIdT>(value);
  }
  template <typename StatementIdT = Aws::String>
  RemovePermissionRequest& WithStatementId(StatementIdT&& value) {
    SetStatementId(std::forward<StatementIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies whether to remove all permissions.</p>
   */
  inline bool GetRemoveAllPermissions() const { return m_removeAllPermissions; }
  inline bool RemoveAllPermissionsHasBeenSet() const { return m_removeAllPermissionsHasBeenSet; }
  inline void SetRemoveAllPermissions(bool value) {
    m_removeAllPermissionsHasBeenSet = true;
    m_removeAllPermissions = value;
  }
  inline RemovePermissionRequest& WithRemoveAllPermissions(bool value) {
    SetRemoveAllPermissions(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the event bus to revoke permissions for. If you omit this, the
   * default event bus is used.</p>
   */
  inline const Aws::String& GetEventBusName() const { return m_eventBusName; }
  inline bool EventBusNameHasBeenSet() const { return m_eventBusNameHasBeenSet; }
  template <typename EventBusNameT = Aws::String>
  void SetEventBusName(EventBusNameT&& value) {
    m_eventBusNameHasBeenSet = true;
    m_eventBusName = std::forward<EventBusNameT>(value);
  }
  template <typename EventBusNameT = Aws::String>
  RemovePermissionRequest& WithEventBusName(EventBusNameT&& value) {
    SetEventBusName(std::forward<EventBusNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_statementId;

  bool m_removeAllPermissions{false};

  Aws::String m_eventBusName;
  bool m_statementIdHasBeenSet = false;
  bool m_removeAllPermissionsHasBeenSet = false;
  bool m_eventBusNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudWatchEvents
}  // namespace Aws
